// Admin JavaScript for PHP version

// Login function
async function login(event) {
    event.preventDefault();
    
    const formData = new FormData();
    formData.append('action', 'login');
    formData.append('username', document.getElementById('username').value);
    formData.append('password', document.getElementById('password').value);
    
    try {
        const response = await fetch('admin.php', {
            method: 'POST',
            body: formData
        });
        
        const result = await response.json();
        
        if (result.success) {
            location.reload();
        } else {
            alert(result.message || 'Login failed');
        }
    } catch (error) {
        alert('Login error');
    }
}

// Show specific admin page
function showPage(page) {
    document.querySelectorAll('.admin-page').forEach(p => p.style.display = 'none');
    document.querySelectorAll('.nav-btn').forEach(btn => btn.classList.remove('active'));
    
    document.getElementById(`${page}-page`).style.display = 'block';
    document.querySelector(`[onclick="showPage('${page}')"]`).classList.add('active');
    
    if (page === 'dashboard') loadDashboard();
    if (page === 'categories') loadCategories();
    if (page === 'items') loadItems();
    if (page === 'settings') loadSettings();
}

// Load dashboard stats
async function loadDashboard() {
    try {
        const response = await fetch('api.php?action=get_stats');
        const stats = await response.json();
        
        document.getElementById('total-items').textContent = stats.total;
        document.getElementById('food-items').textContent = stats.food;
        document.getElementById('drink-items').textContent = stats.drinks;
    } catch (error) {
        console.error('Error loading stats:', error);
    }
}

// Update restaurant name
async function updateRestaurantName(event) {
    event.preventDefault();
    
    const name = document.getElementById('restaurant-name').value;
    
    const formData = new FormData();
    formData.append('action', 'update_setting');
    formData.append('key', 'restaurant_name');
    formData.append('value', name);
    
    try {
        const response = await fetch('api.php', {
            method: 'POST',
            body: formData
        });
        
        const result = await response.json();
        
        if (result.success) {
            alert('Restaurant name updated successfully!');
            document.getElementById('restaurant-name').value = '';
        }
    } catch (error) {
        alert('Update failed');
    }
}

// Load settings
async function loadSettings() {
    try {
        const response = await fetch('api.php?action=get_setting&key=restaurant_name');
        const result = await response.json();
        
        if (result.value) {
            document.getElementById('restaurant-name').value = result.value;
        }
    } catch (error) {
        console.error('Error loading settings:', error);
    }
}

// Add category
async function addCategory(event) {
    event.preventDefault();
    
    const formData = new FormData();
    formData.append('action', 'add_category');
    formData.append('name', document.getElementById('category-name').value);
    formData.append('type', document.getElementById('category-type').value);
    
    try {
        const response = await fetch('api.php', {
            method: 'POST',
            body: formData
        });
        
        const result = await response.json();
        
        if (result.success) {
            document.getElementById('category-name').value = '';
            document.getElementById('category-type').value = '';
            loadCategories();
            loadItems();
        }
    } catch (error) {
        alert('Add category failed');
    }
}

// Load categories
async function loadCategories() {
    try {
        const response = await fetch('api.php?action=get_categories');
        const categories = await response.json();
        
        const container = document.getElementById('categories-display');
        container.innerHTML = categories.map((category, index) => `
            <div class="category-item">
                <div class="category-info">
                    <h4>${category.name}</h4>
                    <p>Type: ${category.type} | Key: ${category.category_key}</p>
                </div>
                <div class="category-actions">
                    <button class="order-btn" onclick="moveCategory(${category.id}, 'up')">↑</button>
                    <button class="order-btn" onclick="moveCategory(${category.id}, 'down')">↓</button>
                    <button onclick="deleteCategory(${category.id})">Delete</button>
                </div>
            </div>
        `).join('');
    } catch (error) {
        console.error('Error loading categories:', error);
    }
}

// Move category
async function moveCategory(id, direction) {
    const formData = new FormData();
    formData.append('action', 'move_category');
    formData.append('id', id);
    formData.append('direction', direction);
    
    try {
        const response = await fetch('api.php', {
            method: 'POST',
            body: formData
        });
        
        const result = await response.json();
        
        if (result.success) {
            loadCategories();
        }
    } catch (error) {
        alert('Move failed');
    }
}

// Delete category
async function deleteCategory(id) {
    if (!confirm('Are you sure you want to delete this category?')) return;
    
    const formData = new FormData();
    formData.append('action', 'delete_category');
    formData.append('id', id);
    
    try {
        const response = await fetch('api.php', {
            method: 'POST',
            body: formData
        });
        
        const result = await response.json();
        
        if (result.success) {
            loadCategories();
            loadItems();
        }
    } catch (error) {
        alert('Delete failed');
    }
}

// Load items
async function loadItems() {
    try {
        const [categoriesResponse, itemsResponse] = await Promise.all([
            fetch('api.php?action=get_categories'),
            fetch('api.php?action=get_items')
        ]);
        
        const categories = await categoriesResponse.json();
        const items = await itemsResponse.json();
        
        // Update category dropdowns
        const categoryOptions = '<option value="">Select Category</option>' +
            categories.map(cat => `<option value="${cat.category_key}">${cat.name}</option>`).join('');
        
        document.getElementById('item-category').innerHTML = categoryOptions;
        document.getElementById('edit-item-category').innerHTML = categoryOptions;
        
        // Display items
        const container = document.getElementById('admin-items');
        container.innerHTML = items.map(item => `
            <div class="admin-item">
                <div class="admin-item-info">
                    <h4>${item.name}</h4>
                    <p>${item.description}</p>
                    <p><strong>Price:</strong> ₦${parseFloat(item.price).toFixed(2)} | <strong>Category:</strong> ${item.category_key}</p>
                </div>
                <div class="admin-item-actions">
                    <button class="edit-btn" onclick="openEditPopup(${item.id}, '${item.name}', ${item.price}, '${item.category_key}')">Edit</button>
                    <button class="delete-btn" onclick="deleteItem(${item.id})">Delete</button>
                </div>
            </div>
        `).join('');
    } catch (error) {
        console.error('Error loading items:', error);
    }
}

// Add item
async function addItem(event) {
    event.preventDefault();
    
    const formData = new FormData();
    formData.append('action', 'add_item');
    formData.append('name', document.getElementById('item-name').value);
    formData.append('description', document.getElementById('item-description').value);
    formData.append('price', document.getElementById('item-price').value);
    formData.append('category', document.getElementById('item-category').value);
    
    try {
        const response = await fetch('api.php', {
            method: 'POST',
            body: formData
        });
        
        const result = await response.json();
        
        if (result.success) {
            document.getElementById('item-name').value = '';
            document.getElementById('item-description').value = '';
            document.getElementById('item-price').value = '';
            document.getElementById('item-category').value = '';
            loadItems();
            loadDashboard();
        }
    } catch (error) {
        alert('Add item failed');
    }
}

// Open edit popup
function openEditPopup(id, name, price, category) {
    document.getElementById('edit-item-id').value = id;
    document.getElementById('edit-item-name').value = name;
    document.getElementById('edit-item-price').value = price;
    document.getElementById('edit-item-category').value = category;
    document.getElementById('edit-popup').style.display = 'flex';
}

// Close edit popup
function closeEditPopup() {
    document.getElementById('edit-popup').style.display = 'none';
}

// Update item
async function updateItem(event) {
    event.preventDefault();
    
    const formData = new FormData();
    formData.append('action', 'update_item');
    formData.append('id', document.getElementById('edit-item-id').value);
    formData.append('name', document.getElementById('edit-item-name').value);
    formData.append('price', document.getElementById('edit-item-price').value);
    formData.append('category', document.getElementById('edit-item-category').value);
    
    try {
        const response = await fetch('api.php', {
            method: 'POST',
            body: formData
        });
        
        const result = await response.json();
        
        if (result.success) {
            closeEditPopup();
            loadItems();
            loadDashboard();
            alert('Item updated successfully!');
        }
    } catch (error) {
        alert('Update failed');
    }
}

// Delete item
async function deleteItem(id) {
    if (!confirm('Are you sure you want to delete this item?')) return;
    
    const formData = new FormData();
    formData.append('action', 'delete_item');
    formData.append('id', id);
    
    try {
        const response = await fetch('api.php', {
            method: 'POST',
            body: formData
        });
        
        const result = await response.json();
        
        if (result.success) {
            loadItems();
            loadDashboard();
        }
    } catch (error) {
        alert('Delete failed');
    }
}

// Initialize
document.addEventListener('DOMContentLoaded', function() {
    if (document.getElementById('admin-panel')) {
        showPage('dashboard');
    }
});
